package com.ewaytek.deepseek.common.bean.base;

import com.ewaytek.deepseek.common.enums.ErrorCode;
import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.io.Serializable;
import java.util.Objects;

/**
 * 一般http响应对象
 *
 * @author lhx
 * @Date 2023/1/23 21:23
 */
@Data
@Slf4j
@NoArgsConstructor
@AllArgsConstructor
//从对象的字段生成hashCode和equals实现，callSuper = false指的是不调用父类的方法
@EqualsAndHashCode(callSuper = false)
//@ApiModel(value = "响应对象（单数据）", description = "响应对象（单数据）")
public class ApiResponse<T> implements Serializable {
    /**
     *
     */
    private static final long serialVersionUID = 1L;
    /**
     * 错误码前缀
     */
    public static int prefixErrorCode;
    /**
     * 响应编码：0成功；-1系统异常；
     */
//    @ApiModelProperty(value = "响应编码", notes = "0成功；其他：错误，详见错误编码")
    private int code;
    /**
     * 响应结果描述
     */
//    @ApiModelProperty(value = "错误描述", notes = "错误描述")
    private String msg;
    /**
     * 业务数据
     */
//    @ApiModelProperty(value = "业务数据")
    private T data;

    private String trace;

    /**
     * 无业务数据的成功响应
     *
     * @return
     */
    public static <T> ApiResponse<T> success() {
        return success(null);
    }

    /**
     * 带业务数据的成功响应
     *
     * @param data
     * @param <T>
     * @return
     */
    public static <T> ApiResponse<T> success(T data) {
        return success(data, ErrorCode.SUCCESS.getErrcode(), ErrorCode.SUCCESS.getErrmsg());
    }

    @JsonIgnore
    public boolean belongSuccess() {
        return Objects.equals(ErrorCode.SUCCESS.getErrcode(), this.code);
    }

    @JsonIgnore
    public boolean belongError() {
        return !Objects.equals(ErrorCode.SUCCESS.getErrcode(), this.code);
    }

    /**
     * 带业务数据的成功响应
     *
     * @param data
     * @param <T>
     * @return
     */
    public static <T> ApiResponse<T> success(T data, String msg) {
        return success(data, ErrorCode.SUCCESS.getErrcode(), msg);
    }

    /**
     * 带业务数据的成功响应
     *
     * @param data
     * @param <T>
     * @return
     */
    public static <T> ApiResponse<T> success(T data, int code, String msg) {
        ApiResponse<T> response = new ApiResponse<T>();
        response.setCode(code);
        response.setMsg(msg);
        response.setData(data);
        return response;
    }

    /**
     * 响应失败
     *
     * @return
     */
    public static <T> ApiResponse<T> fail() {
        return fail(ErrorCode.INTERNAL_ERR.getErrcode(), ErrorCode.INTERNAL_ERR.getErrmsg());
    }

    /**
     * 响应失败
     *
     * @param responseCode
     * @return
     */
    public static <T> ApiResponse<T> fail(ErrorCode responseCode) {
        return fail(responseCode.getErrcode(), responseCode.getErrmsg());
    }

    /**
     * 响应失败
     *
     * @param msg
     * @return
     */
    public static <T> ApiResponse<T> fail(String msg) {
        return fail(ErrorCode.FAIL.getErrcode(), msg);
    }

    /**
     * 响应失败
     *
     * @param failCode
     * @param msg
     * @return
     */
    public static <T> ApiResponse<T> fail(int failCode, String msg) {
        ApiResponse<T> response = new ApiResponse<T>();
        response.setCode(prefixErrorCode + failCode);
        response.setMsg(msg);
        log.info("响应失败，返回报文为{}", response.toString());
        return response;
    }

    /**
     * 响应失败
     *
     * @param data     响应出错的数据
     * @param failCode
     * @param msg
     * @return
     */
    public static <T> ApiResponse<T> fail(T data, int failCode, String msg) {
        ApiResponse<T> response = new ApiResponse<T>();
        response.setCode(prefixErrorCode + failCode);
        response.setMsg(msg);
        response.setData(data);
        log.info("响应失败，返回报文为{}", response.toString());
        return response;
    }
}
