package com.ewaytek.deepseek.service.dify.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.ContentType;
import cn.hutool.http.Header;
import com.alibaba.fastjson2.JSON;
import com.ewaytek.deepseek.common.exception.base.BaseException;
import com.ewaytek.deepseek.doadmin.dto.dify.DifyChatBlockIngDTO;
import com.ewaytek.deepseek.doadmin.dto.dify.DifyResponseDTO;
import com.ewaytek.deepseek.doadmin.dto.doubao.DoubaoChatDTO;
import com.ewaytek.deepseek.doadmin.vo.dify.DifyBlockingVO;
import com.ewaytek.deepseek.doadmin.vo.dify.DifyChatVO;
import com.ewaytek.deepseek.service.dify.DifyApi;
import com.ewaytek.deepseek.service.dify.KeyStrategyFunction;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.sse.EventSource;
import okhttp3.sse.EventSourceListener;
import okhttp3.sse.EventSources;
import org.jetbrains.annotations.NotNull;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory;
import retrofit2.converter.jackson.JacksonConverterFactory;
import retrofit2.http.Body;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * @author yangtq
 * @date 2025/2/25
 */


@Slf4j
public class DifyApiClient {

    //自定义api host使用builder的方式构造client
    @Getter
    private String apiHost;
    @Getter
    private String apiKey;
    @Getter
    private DifyApi difyApi;

    // 自定义okHttpClient,非自定义为sdk默认OkHttpClient实例
    @Getter
    private OkHttpClient okHttpClient;

    // api key的获取策略
    @Getter
    private KeyStrategyFunction<List<String>, String> keyStrategy;


    /**
     * 构造器
     *
     * @return OpenAiClient.Builder
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * 构造
     *
     * @param builder
     */
    private DifyApiClient(Builder builder) {
        if (StrUtil.isBlank(builder.apiHost)) {
            builder.apiHost = apiHost;
        }
        apiHost = builder.apiHost;
        apiKey = builder.apiKey;
        if (Objects.isNull(builder.okHttpClient)) {
            builder.okHttpClient = this.okHttpClient();
        } else {
            //自定义的okhttpClient  需要增加api keys
            builder.okHttpClient = builder.okHttpClient
                    .newBuilder()
                    .build();
        }
        okHttpClient = builder.okHttpClient;
        this.difyApi = new Retrofit.Builder()
                .baseUrl(apiHost)
                .client(okHttpClient)
                .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create())
                .build().create(DifyApi.class);
    }

    /**
     * 创建默认OkHttpClient
     *
     * @return
     */
    private OkHttpClient okHttpClient() {
        return new OkHttpClient
                .Builder()
                .connectTimeout(30, TimeUnit.SECONDS)
                .writeTimeout(30, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS).build();
    }

    public static final class Builder {
        //api keys
        private @NotNull String apiKey;
        //api请求地址，结尾处有斜杠
        private String apiHost;
        //自定义OkhttpClient
        private OkHttpClient okHttpClient;
        // api key的获取策略
        private KeyStrategyFunction keyStrategy;

        public Builder() {
        }

        /**
         * @param val api请求地址，结尾处有斜杠
         * @return Builder对象
         */
        public Builder apiHost(String val) {
            apiHost = val;
            return this;
        }

        public Builder apiKey(@NotNull String val) {
            apiKey = val;
            return this;
        }

        public Builder keyStrategy(KeyStrategyFunction val) {
            keyStrategy = val;
            return this;
        }

        public Builder okHttpClient(OkHttpClient val) {
            okHttpClient = val;
            return this;
        }

        public DifyApiClient build() {
            return new DifyApiClient(this);
        }
    }


    /**
     * 流式输出
     *
     * @param difyRequest
     * @param eventSourceListener
     * @param <T>
     */
    public <T extends DifyResponseDTO> void streamChatCompletion(DifyChatVO difyRequest, EventSourceListener eventSourceListener) {
        if (Objects.isNull(eventSourceListener)) {
            log.info("EventSourceListener为空");
            throw new RuntimeException("EventSourceListener为空");
        }
        try {
            if (CollectionUtil.isNotEmpty(difyRequest.getInputs())) {
                difyRequest.setInputs(new HashMap<>());
            }
            if (CollectionUtil.isNotEmpty(difyRequest.getFiles())) {
                difyRequest.setFiles(new ArrayList<>());
            }
            //构建请求参数json数据
            ObjectMapper mapper = new ObjectMapper();
            String requestBody = mapper.writeValueAsString(difyRequest);
            log.info("请求参数：{}", requestBody);
            //创建事件工厂
            EventSource.Factory factory = EventSources.createFactory(this.okHttpClient);

            Request request = new Request.Builder()
                    .url(this.apiHost + "chat-messages")
                    .addHeader(Header.AUTHORIZATION.getValue(), "Bearer " + apiKey)
                    .addHeader(Header.ACCEPT.getValue(),ContentType.EVENT_STREAM.getValue())
                    .post(RequestBody.create(MediaType.parse(ContentType.JSON.getValue()), requestBody))
                    .build();
            factory.newEventSource(request, eventSourceListener);
        } catch (Exception e) {
            log.error("请求参数解析异常：{}", e.getMessage());
        }
    }


    /**
     * 流式输出
     *
     * @param doubaoChatDTO
     * @param eventSourceListener
     * @param <T>
     */
    public <T extends DifyResponseDTO> void streamChatCompletionDoubao(DoubaoChatDTO doubaoChatDTO, EventSourceListener eventSourceListener) {
        if (Objects.isNull(eventSourceListener)) {
            log.info("EventSourceListener为空");
            throw new RuntimeException("EventSourceListener为空");
        }
        try {
        DifyChatBlockIngDTO difyChatDTO = new DifyChatBlockIngDTO();
        difyChatDTO.setQuery(doubaoChatDTO.getModel());
        difyChatDTO.setUser("ewaytek" + System.currentTimeMillis());
        difyChatDTO.setResponseMode("streaming");
            //构建请求参数json数据
            ObjectMapper mapper = new ObjectMapper();
            String requestBody = mapper.writeValueAsString(difyChatDTO);
            log.info("请求参数：{}", requestBody);
            //创建事件工厂
            EventSource.Factory factory = EventSources.createFactory(this.okHttpClient);

            Request request = new Request.Builder()
                    .url(this.apiHost + "chat-messages")
                    .addHeader(Header.AUTHORIZATION.getValue(), "Bearer " + "app-Lrj9iXzzJN8dXG3DvldKVVCB")
                    .addHeader(Header.ACCEPT.getValue(),ContentType.EVENT_STREAM.getValue())
                    .post(RequestBody.create(MediaType.parse(ContentType.JSON.getValue()), requestBody))
                    .build();
            factory.newEventSource(request, eventSourceListener);
        } catch (Exception e) {
            log.error("请求参数解析异常：{}", e.getMessage());
        }
    }

    /**
     * 阻塞式问答
     *
     * @param difyRequest chat completion
     * @return 返回答案
     */
    public DifyBlockingVO chatMessages(@Body DifyChatVO difyRequest, String serverKey) {
        if (difyRequest.getInputs() == null) {
            difyRequest.setInputs(new HashMap<>());
        }
        if (difyRequest.getFiles() == null) {
            difyRequest.setFiles(new ArrayList<>());
        }
        log.debug(JSON.toJSONString(difyRequest));

        // 序列化请求体
        ObjectMapper mapper = new ObjectMapper();
        String requestBodyJson = "";
        try {
            requestBodyJson = mapper.writeValueAsString(difyRequest);
        } catch (Exception e) {
            log.error("请求体序列化失败：{}", e.getMessage());
            throw new RuntimeException(e.getMessage());
        }

        // 创建请求体
        RequestBody requestBody = RequestBody.create(MediaType.parse(ContentType.JSON.getValue()), requestBodyJson);

        // 创建请求对象，这里动态地将API Key设置到请求头中
        Request request = new Request.Builder()
                .url(this.apiHost + "chat-messages")  // 此处路径根据实际需要进行调整
                .addHeader("Authorization", "Bearer " + serverKey) // 设置动态API Key
                .post(requestBody)
                .build();

        DifyBlockingVO response;
        try {
            // 执行同步请求并获取响应
            okhttp3.Response okHttpResponse = okHttpClient.newCall(request).execute();
            if (!okHttpResponse.isSuccessful() || okHttpResponse.body() == null) {
                log.error("请求失败：HTTP {},message: {}", okHttpResponse.code(), okHttpResponse.message());
                throw new BaseException("请求失败：HTTP " + okHttpResponse.code() + " " + okHttpResponse.message());
            }

            // 反序列化响应体
            String responseBody = okHttpResponse.body().string();
            response = mapper.readValue(responseBody, DifyBlockingVO.class);
        } catch (Exception e) {
            log.error("请求异常：{}", e.getMessage());
            throw new RuntimeException(e.getMessage());
        }

        // 返回结果
        return response;
    }
}
