package com.ewaytek.deepseek.service.dify.impl;

import cn.hutool.cache.impl.TimedCache;
import cn.hutool.core.date.DateUnit;
import cn.hutool.http.ContentType;
import cn.hutool.http.Header;
import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpResponse;
import com.alibaba.fastjson2.JSON;
import com.ewaytek.deepseek.common.bean.base.ApiResponse;
import com.ewaytek.deepseek.common.utils.poi.ExcelUtil;
import com.ewaytek.deepseek.config.DifyConfig;
import com.ewaytek.deepseek.config.DifySseEventSourceListener;
import com.ewaytek.deepseek.doadmin.dto.dify.DIfyImportVerifyDTO;
import com.ewaytek.deepseek.doadmin.dto.dify.DifyChatBlockIngDTO;
import com.ewaytek.deepseek.doadmin.dto.dify.DifyChatDTO;
import com.ewaytek.deepseek.doadmin.vo.dify.*;
import com.ewaytek.deepseek.service.dify.DifyChatService;
import com.ewaytek.deepseek.task.DifyThread;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.util.concurrent.RateLimiter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyEmitter;
import reactor.core.publisher.Flux;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;


/**
 * @author yangtq
 * @date 2025/2/25
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class DifyChatServiceImpl implements DifyChatService {

    private final RateLimiter rateLimiter = RateLimiter.create(5.0);

    /**
     * 设置sse链接时长缓存
     */
    public static final long TIMEOUT = 30 * DateUnit.MINUTE.getMillis();

    public static final TimedCache<String, Object> LOCAL_CACHE = new TimedCache<>(TIMEOUT);

    @Resource
    private DifyApiClient difyApiClient;

    @Resource
    private DifyConfig difyConfig;

    @Resource
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;

    @Autowired
    private OkHttpClient httpClient;

    @Override
    public ResponseBodyEmitter sseChatPrompt(DifyChatVO difyChatVO) {
        ResponseBodyEmitter sseEmitter = this.getResponseBodyEmitter(difyChatVO);
        DifySseEventSourceListener listener = new DifySseEventSourceListener(sseEmitter);
        difyApiClient.streamChatCompletion(difyChatVO, listener);
        return sseEmitter;
    }


    @Override
    public ApiResponse<List<ConversationsVO.ConversationInfoVO>> conversations(Integer limit, Boolean pinned) {
        String url = String.format("%s?limit=%d&pinned=%s",
                difyConfig.getConversationUrl(),
                limit,
                pinned);

        HttpRequest request = HttpRequest.get(url)
                .header(Header.AUTHORIZATION, "Bearer " + difyConfig.getApiKey())
                .header(Header.CONTENT_TYPE, "application/json")
                .timeout(60000);
        try (HttpResponse response = request.execute()) {
            if (response.isOk()) {
                String responseBody = response.body();
                log.info("Response: {}", responseBody);
                ConversationsVO conversationsVO = JSON.parseObject(responseBody, ConversationsVO.class);
                return ApiResponse.success(conversationsVO.getData());
            } else {
                log.error("Request failed with status code: {}", response.getStatus());
                return ApiResponse.fail("Request failed with status code: " + response.getStatus());
            }
        } catch (Exception e) {
            log.error("Error during request: ", e);
            return ApiResponse.fail("Error during request: " + e.getMessage());
        }
    }


    @Override
    public void importVerify2(HttpServletRequest request, HttpServletResponse response, MultipartFile file) throws IOException, InterruptedException {
        ExcelUtil<DIfyImportVerifyDTO> util = new ExcelUtil<DIfyImportVerifyDTO>(DIfyImportVerifyDTO.class);
        List<DIfyImportVerifyDTO> list = util.importExcel(file.getInputStream());
        if (!CollectionUtils.isEmpty(list)) {
            // 动态设置线程池大小
            int corePoolSize = Runtime.getRuntime().availableProcessors();
            int maximumPoolSize = corePoolSize * 2;
            long keepAliveTime = 60L;
            TimeUnit unit = TimeUnit.SECONDS;
            BlockingQueue<Runnable> workQueue = new LinkedBlockingQueue<>(100);
            ExecutorService pool = new ThreadPoolExecutor(
                    corePoolSize,
                    maximumPoolSize,
                    keepAliveTime,
                    unit,
                    workQueue
            );

            int batchSize = 10; // 每批次提交的任务数
            int totalTasks = list.size();
            List<DIfyImportVerifyDTO> results = new ArrayList<>();
            for (int i = 0; i < totalTasks; i += batchSize) {
                int end = Math.min(i + batchSize, totalTasks);
                List<DIfyImportVerifyDTO> batch = list.subList(i, end);
                List<Callable<DIfyImportVerifyDTO>> tasks = batch.stream()
                        .map(dto -> (Callable<DIfyImportVerifyDTO>) () -> processBatch(dto))
                        .collect(Collectors.toList());

                List<Future<DIfyImportVerifyDTO>> futures = pool.invokeAll(tasks);
                for (Future<DIfyImportVerifyDTO> future : futures) {
                    try {
                        results.add(future.get());
                    } catch (InterruptedException | ExecutionException e) {
                        log.error("Error processing batch", e);
                    }
                }
            }

            util.exportExcel(results, "dify");
            // 导出结果到 Excel
            util.exportExcel(response, results, "dify");

            // 关闭线程池
            pool.shutdown();

        }
    }

    public DIfyImportVerifyDTO processBatch(DIfyImportVerifyDTO dto) throws JsonProcessingException {
        DifyChatBlockIngDTO difyChatDTO = new DifyChatBlockIngDTO();
        difyChatDTO.setQuery(dto.getQuestion());
        difyChatDTO.setUser("ewaytek" + System.currentTimeMillis());
        difyChatDTO.setResponseMode("streaming");
        //构建请求参数json数据
        ObjectMapper mapper = new ObjectMapper();
        String requestBody = mapper.writeValueAsString(difyChatDTO);
        Headers headers = new Headers.Builder().add("Authorization", "Bearer " + difyConfig.getApiKey()).add("Content-Type", "application/json").build();

        Request request = new Request.Builder().url(difyConfig.getApiHost() + "chat-messages").post(RequestBody.create(MediaType.parse(ContentType.JSON.getValue()), requestBody))
                .headers(headers).build();
        List<RetrieverResources> retrieverResourcesList = new ArrayList<>();

        try (Response response = httpClient.newCall(request).execute()) {
            if (!response.isSuccessful()) {

                return dto;
            }
            // 处理流式响应
            ResponseBody responseBody = response.body();
            if (responseBody != null) {
                StringBuilder responseBuilder = new StringBuilder();

                while (!responseBody.source().exhausted()) {
                    String line = responseBody.source().readUtf8Line();
                    if (line != null && !line.isEmpty()) {
                        if (line.startsWith("data:")) { // 处理 SSE 格式
                            String eventData = line.substring(5).trim(); // 去掉 "data:" 前缀
                            DifyStreamVO blockingVO = JSON.parseObject(eventData, DifyStreamVO.class);
                            MetadataVO metadataVO = blockingVO.getMetadata();
                            if (metadataVO != null) {
                                if (!CollectionUtils.isEmpty(metadataVO.getRetrieverResources())) {
                                    retrieverResourcesList.addAll(metadataVO.getRetrieverResources());
                                }
                            }
                            responseBuilder.append(blockingVO.getAnswer());
                        }
                    }
                }
                dto.setRetrieverResources(retrieverResourcesList);
                dto.setAnswer(responseBuilder.toString());
            }

        } catch (IOException e) {
            log.error(e.getMessage());
        }
        return dto;
    }

    @Override
    public ApiResponse importVerify3(MultipartFile file) throws IOException, InterruptedException, ExecutionException {
        ExcelUtil<DIfyImportVerifyDTO> util = new ExcelUtil<>(DIfyImportVerifyDTO.class);
        List<DIfyImportVerifyDTO> dataList = util.importExcel(file.getInputStream());

        if (!CollectionUtils.isEmpty(dataList)) {

            for (DIfyImportVerifyDTO ifyImportVerifyDTO : dataList) {
                processBatch(ifyImportVerifyDTO);
            }
            return util.exportExcel(dataList, "import_verify_results");
        }
        return ApiResponse.fail();
    }




    @Override
    public ApiResponse importVerify(MultipartFile file) throws IOException, InterruptedException, ExecutionException {
        ExcelUtil<DIfyImportVerifyDTO> util = new ExcelUtil<>(DIfyImportVerifyDTO.class);
        List<DIfyImportVerifyDTO> dataList = util.importExcel(file.getInputStream());

        if (!CollectionUtils.isEmpty(dataList)) {
            // 提交任务到线程池
            List<CompletableFuture<DIfyImportVerifyDTO>> futures = dataList.stream()
                    .map(data -> CompletableFuture.supplyAsync(() -> {
                        DifyThread thread = new DifyThread(data, difyConfig, httpClient, null);
                        thread.run(); // 执行任务
                        return data; // 返回处理后的数据
                    }, threadPoolTaskExecutor))
                    .collect(Collectors.toList());

            // 等待所有任务完成
            CompletableFuture<Void> allFutures = CompletableFuture.allOf(
                    futures.toArray(new CompletableFuture[0]));

            allFutures.get(); // 阻塞等待所有任务完成

            // 收集所有结果
            List<DIfyImportVerifyDTO> results = futures.stream()
                    .map(CompletableFuture::join)
                    .collect(Collectors.toList());

            // 导出结果到 Excel 文件
            return util.exportExcel(results, "import_verify_results");
        }
        return ApiResponse.fail();
    }


    /**
     * 创建sse连接
     *
     * @param chatRequest
     * @return
     */
    private ResponseBodyEmitter getResponseBodyEmitter(DifyChatVO chatRequest) {
        //0L设置允许超时
        ResponseBodyEmitter sseEmitter = new ResponseBodyEmitter(0L);
        sseEmitter.onCompletion(() -> {
            log.info("会话[{}]sse结束连接......", chatRequest.getConversationId());
            LOCAL_CACHE.remove(chatRequest.getConversationId());
        });
        //超时回调
        sseEmitter.onTimeout(() -> {
            log.error("会话[{}]sse连接超时......", chatRequest.getConversationId());
        });
        //异常回调
        sseEmitter.onError(
                throwable -> {
                    log.error("会话[{}]sse连接失败......", chatRequest.getConversationId());
                }
        );
        LOCAL_CACHE.put(chatRequest.getConversationId(), sseEmitter);
        log.info("会话[{}]创建sse连接成功！", chatRequest.getConversationId());
        return sseEmitter;
    }
}
