package com.ewaytek.deepseek.task;

import cn.hutool.http.ContentType;
import com.alibaba.fastjson2.JSON;
import com.ewaytek.deepseek.config.DifyConfig;
import com.ewaytek.deepseek.doadmin.dto.dify.DIfyImportVerifyDTO;
import com.ewaytek.deepseek.doadmin.dto.dify.DifyChatBlockIngDTO;
import com.ewaytek.deepseek.doadmin.dto.dify.demo.DIfyWorkflowsResultDTO;
import com.ewaytek.deepseek.doadmin.dto.dify.demo.DifyWorkflowsDTO;
import com.ewaytek.deepseek.doadmin.vo.dify.DifyStreamVO;
import com.ewaytek.deepseek.doadmin.vo.dify.MetadataVO;
import com.ewaytek.deepseek.doadmin.vo.dify.RetrieverResources;
import com.ewaytek.deepseek.doadmin.vo.dify.workflows.WorkflowsExecVO;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.util.concurrent.RateLimiter;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.collections4.CollectionUtils;

import java.io.IOException;
import java.text.DecimalFormat;
import java.util.*;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author yangtq
 * @date 2025/3/6
 */
@Slf4j
public class DifyWorkflowsThread implements Runnable {

    private DifyConfig difyConfig;
    private static final RateLimiter rateLimiter = RateLimiter.create(5.0); // 每秒最多 5 次调用
    private WorkflowsExecVO workflowsExecVO; // 用于存储处理结果

    private OkHttpClient httpClient;

    private static final List<DIfyImportVerifyDTO> results = Collections.synchronizedList(new ArrayList<>());
    private DIfyImportVerifyDTO data;

    public DifyWorkflowsThread(DIfyImportVerifyDTO data, DifyConfig difyConfig, OkHttpClient httpClient) {
        this.data = data;
        this.difyConfig = difyConfig;
        this.httpClient = httpClient;
    }

    // 提供一个方法来获取处理后的 WorkflowsExecVO 对象
    public WorkflowsExecVO getWorkflowsExecVO() {
        return workflowsExecVO;
    }

    @Override
    public void run() {
        try {
            rateLimiter.acquire(); // 获取令牌，阻塞等待直到获取成功
            // 调用大模型处理数据
             workflowsExecVO = new WorkflowsExecVO();
            DIfyImportVerifyDTO result = processBatch(data);
            workflowsExecVO.setQuestion(data.getQuestion());
            workflowsExecVO.setStandard(data.getStandard());
            workflowsExecVO.setAnswer(result.getAnswer());
            String answer = cleanContent(result.getAnswer());
            workflowsExecVO.setQuestion1(answer);
            workflowsExecVO.setTime(result.getTime());
            workflowsExecVO = DifyWorkflows(workflowsExecVO);

            // 处理结果
            System.out.println("Processed result: " + result.getAnswer());
        } catch (Exception e) {
            System.err.println("Error processing data: " + e.getMessage());
        }
    }


    private DIfyImportVerifyDTO processBatch(DIfyImportVerifyDTO dto) throws Exception {
        DifyChatBlockIngDTO difyChatDTO = new DifyChatBlockIngDTO();
        difyChatDTO.setQuery(dto.getQuestion());
        difyChatDTO.setUser("ewaytek");
        difyChatDTO.setResponseMode("streaming");
        //构建请求参数json数据
        ObjectMapper mapper = new ObjectMapper();
        String requestBody = mapper.writeValueAsString(difyChatDTO);
        Headers headers = new Headers.Builder().add("Authorization", "Bearer " + "app-9D2IddTOvZDeBGLm1iNtQ0EF").add("Content-Type", "application/json").build();

        Request request = new Request.Builder().url(difyConfig.getApiHost() + "chat-messages").post(RequestBody.create(MediaType.parse(ContentType.JSON.getValue()), requestBody))
                .headers(headers).build();
        List<RetrieverResources> retrieverResourcesList = new ArrayList<>();

        Date start = new Date();
        try (Response response = httpClient.newCall(request).execute()) {
            if (!response.isSuccessful()) {

                return dto;
            }
            // 处理流式响应
            ResponseBody responseBody = response.body();
            if (responseBody != null) {
                StringBuilder responseBuilder = new StringBuilder();

                while (!responseBody.source().exhausted()) {
                    String line = responseBody.source().readUtf8Line();
                    if (line != null && !line.isEmpty()) {
                        if (line.startsWith("data:")) { // 处理 SSE 格式
                            String eventData = line.substring(5).trim(); // 去掉 "data:" 前缀
                            DifyStreamVO blockingVO = JSON.parseObject(eventData, DifyStreamVO.class);
                            MetadataVO metadataVO = blockingVO.getMetadata();
                            if (metadataVO != null) {
                                if (!CollectionUtils.isEmpty(metadataVO.getRetrieverResources())) {
                                    retrieverResourcesList.addAll(metadataVO.getRetrieverResources());
                                }
                            }
                            responseBuilder.append(blockingVO.getAnswer());
                        }
                    }
                }
                Date end = new Date();
                String decimalFormat=   new DecimalFormat("0.0000").format((end.getTime() - start.getTime())/1000);
                dto.setRetrieverResources(retrieverResourcesList);
                dto.setAnswer(responseBuilder.toString());
                dto.setTime(decimalFormat);
            }

        } catch (IOException e) {
            log.error(e.getMessage());
        }
        return dto;
    }

    public WorkflowsExecVO DifyWorkflows(WorkflowsExecVO workflowsExecVO) throws JsonProcessingException {
        DifyWorkflowsDTO difyWorkflowsDTO = new DifyWorkflowsDTO();
        difyWorkflowsDTO.setUser("abc-123");
        difyWorkflowsDTO.setResponseMode("streaming");
        Map<String, String> inputs = new HashMap<>();
        inputs.put("question", workflowsExecVO.getQuestion1());
        difyWorkflowsDTO.setInputs(inputs);

        ObjectMapper mapper = new ObjectMapper();
        String requestBody = mapper.writeValueAsString(difyWorkflowsDTO);
        Headers headers = new Headers.Builder().add("Authorization", "Bearer " + "app-9RuAoErdMXkG4jVN0YNjXekp").add("Content-Type", "application/json").build();

        Request request = new Request.Builder().url(difyConfig.getApiHost() + "workflows/run").post(okhttp3.RequestBody.create(MediaType.parse(ContentType.JSON.getValue()), requestBody))
                .headers(headers).build();

        List<WorkflowsExecVO.Workflows> workflowsList = new ArrayList<>();
        try (Response response = httpClient.newCall(request).execute()) {
            if (!response.isSuccessful()) {
                log.info("DifyWorkflows#" + response.message());
                return workflowsExecVO;
            }
            // 处理流式响应
            ResponseBody responseBody = response.body();
            if (responseBody != null) {

                while (!responseBody.source().exhausted()) {
                    String line = responseBody.source().readUtf8Line();
                    if (line == null || line.isEmpty()) {
                        continue;
                    }
                    if (line.startsWith("data:")) { // 处理 SSE 格式
                        log.info("DifyWorkflows##" + line);
                        String eventData = line.substring(5).trim(); // 去掉 "data:" 前缀
                        DIfyWorkflowsResultDTO blockingVO = JSON.parseObject(eventData, DIfyWorkflowsResultDTO.class);
                        if (!"workflow_finished".equals(blockingVO.getEvent())) {
                            continue;
                        }

                        DIfyWorkflowsResultDTO.Outputs metadataVO = blockingVO.getData().getOutputs();
                        if (metadataVO == null) {
                            continue;
                        }
                        if (CollectionUtils.isEmpty(metadataVO.getResult())) {
                            continue;
                        }

                        for (DIfyWorkflowsResultDTO.ResultItem resultItem : metadataVO.getResult()) {
                            WorkflowsExecVO.Workflows workflows = new WorkflowsExecVO.Workflows();
                            workflows.setReliability(String.valueOf(resultItem.getMetadata().getScore()));
                            workflows.setAnswer(resultItem.getContent());
                            workflowsList.add(workflows);
                        }
                    }
                }
                workflowsExecVO.setWorkflowsList(workflowsList);

            }

        } catch (IOException e) {
            log.error(e.getMessage());
        }
        return workflowsExecVO;
    }

    public static String cleanContent(String content) {
        // 移除 <details> 标签及其内容
        Pattern detailsPattern = Pattern.compile("<details.*?</details>", Pattern.DOTALL);
        Matcher detailsMatcher = detailsPattern.matcher(content);
        content = detailsMatcher.replaceAll("").trim();

        // 移除尾部的 null
        Pattern nullPattern = Pattern.compile("null$", Pattern.DOTALL);
        Matcher nullMatcher = nullPattern.matcher(content);
        content = nullMatcher.replaceAll("").trim();

        // 移除多余的换行符和空格
        Pattern whitespacePattern = Pattern.compile("\\s+\\n", Pattern.DOTALL);
        Matcher whitespaceMatcher = whitespacePattern.matcher(content);
        content = whitespaceMatcher.replaceAll("\n").trim();

        return content;
    }

}
